<?php
session_start();

// Validar sesión
if (!isset($_SESSION["id_usuario"])) {
    header("Location: login.php");
    exit();
}

require 'db.php';

// --- Lógica de Filtros ---
$busqueda = isset($_GET['busqueda']) ? trim($_GET['busqueda']) : "";
$filtro_status = isset($_GET['status']) ? trim($_GET['status']) : "";
$fecha_inicio = isset($_GET['fecha_inicio']) ? trim($_GET['fecha_inicio']) : "";
$fecha_fin = isset($_GET['fecha_fin']) ? trim($_GET['fecha_fin']) : "";

// --- Construcción de la Consulta SQL ---
$sql = "
    SELECT 
        rg.id_recepcion, rg.control, rg.fecha_recepcion, rg.status, 
        rg.aprobado_por, rg.fecha_aprobacion,
        p.razon_social AS proveedor_nombre,
        SUM(dr.peso_neto) AS total_peso_neto
    FROM recepcion_general rg
    LEFT JOIN proveedor p ON rg.rif_proveedor = p.cedula
    LEFT JOIN detalle_recepcion dr ON rg.id_recepcion = dr.id_recepcion
    WHERE 1=1 
";

$params = [];
$types = "";

// Filtro de Búsqueda
if ($busqueda !== "") {
    $sql .= " AND (rg.control LIKE ? OR p.razon_social LIKE ?)";
    $term = "%" . $busqueda . "%";
    $params[] = $term;
    $params[] = $term;
    $types .= "ss";
}

// Filtro de Estatus (¡El que pediste!)
if ($filtro_status !== "") {
    $sql .= " AND rg.status = ?";
    $params[] = $filtro_status;
    $types .= "s";
}

// Filtro de Fecha
if ($fecha_inicio !== "") {
    $sql .= " AND rg.fecha_recepcion >= ?";
    $params[] = $fecha_inicio;
    $types .= "s";
}
if ($fecha_fin !== "") {
    // Agregamos la hora final para incluir todo el día
    $fecha_fin_completa = $fecha_fin . " 23:59:59";
    $sql .= " AND rg.fecha_recepcion <= ?";
    $params[] = $fecha_fin_completa;
    $types .= "s";
}

$sql .= " 
    GROUP BY rg.id_recepcion, rg.control, rg.fecha_recepcion, rg.status, 
             rg.aprobado_por, rg.fecha_aprobacion, p.razon_social
    ORDER BY rg.fecha_recepcion DESC, rg.control DESC
";

$stmt = $conn->prepare($sql);

if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}

$stmt->execute();
$resultado = $stmt->get_result();
$recepciones = $resultado->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$total_reporte = 0; // Para sumar el total
?>

<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Reporte de Recepciones</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    body {
      font-family: 'Segoe UI', sans-serif;
      background-color: #f0fdf4;
      padding: 30px;
    }
    .container {
      max-width: 1400px; /* Más ancho para reportes */
      margin: auto;
      background: #fff;
      padding: 25px 30px;
      border-radius: 15px;
      box-shadow: 0 10px 25px rgba(0, 128, 0, 0.1);
    }
    h2 {
      text-align: center;
      color: #2e7d32;
      margin-bottom: 25px;
    }
    .boton {
      padding: 8px 15px;
      background-color: #2e7d32;
      color: white;
      border: none;
      border-radius: 8px;
      cursor: pointer;
      text-decoration: none;
      display: inline-block;
    }
    .boton:hover {
      background-color: #1b5e20;
      color: white;
    }
    .filtros {
      background-color: #f9fbe7;
      padding: 20px;
      border-radius: 10px;
      border: 1px solid #e0e0e0;
      margin-bottom: 25px;
    }
    table {
      width: 100%;
      border-collapse: collapse;
      margin-top: 25px;
      font-size: 0.95em;
    }
    th, td {
      border: 1px solid #ccc;
      padding: 10px;
      text-align: center;
      vertical-align: middle;
    }
    th {
      background-color: #e8f5e9;
      color: #2e7d32;
    }
    tr:hover {
      background-color: #f1f8e9;
    }
    .total-row {
      background-color: #c8e6c9;
      font-weight: bold;
      color: #1b5e20;
      font-size: 1.05em;
    }
    .badge {
        font-size: 0.9em;
        font-weight: 600;
    }
  </style>
</head>
<body>

<div class="container">
  <h2><i class="bi bi-file-earmark-text"></i> Reporte de Recepciones</h2>

  <form method="GET" class="filtros">
    <div class="row g-3 align-items-end">
        <div class="col-md-3">
            <label for="busqueda" class="form-label">Buscar (Control/Proveedor):</label>
            <input type="text" name="busqueda" id="busqueda" class="form-control" value="<?= htmlspecialchars($busqueda) ?>">
        </div>
        
        <div class="col-md-2">
            <label for="status" class="form-label">Estatus:</label>
            <select name="status" id="status" class="form-select">
                <option value="">-- Todos --</option>
                <option value="APROBADO" <?= $filtro_status == 'APROBADO' ? 'selected' : '' ?>>Aprobado</option>
                <option value="ABIERTO" <?= $filtro_status == 'ABIERTO' ? 'selected' : '' ?>>Abierto</option>
                </select>
        </div>

        <div class="col-md-2">
            <label for="fecha_inicio" class="form-label">Desde:</label>
            <input type="date" name="fecha_inicio" id="fecha_inicio" class="form-control" value="<?= htmlspecialchars($fecha_inicio) ?>">
        </div>
        
        <div class="col-md-2">
            <label for="fecha_fin" class="form-label">Hasta:</label>
            <input type="date" name="fecha_fin" id="fecha_fin" class="form-control" value="<?= htmlspecialchars($fecha_fin) ?>">
        </div>

        <div class="col-md-3 d-flex gap-2">
            <button type="submit" class="boton w-100"><i class="bi bi-search"></i> Buscar</button>
            <a href="reporte_recepciones.php" class="boton w-100" style="background-color: #757575;"><i class="bi bi-arrow-clockwise"></i> Limpiar</a>
            <a href="home.php" class="boton w-100" style="background-color: #1565c0;"><i class="bi bi-house"></i> Volver</a>
        </div>
    </div>
  </form>

  <?php if (count($recepciones) > 0): ?>
      <table>
        <thead>
          <tr>
            <th>Control</th>
            <th>Fecha Recepción</th>
            <th style="width: 25%;">Proveedor</th>
            <th>Estatus</th>
            <th>Total Peso Neto (Kg)</th>
            <th>Aprobado Por</th>
            <th>Fecha Aprobación</th>
            <th>Acción</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($recepciones as $recep): 
              $total_reporte += $recep['total_peso_neto'];
          ?>
            <tr>
              <td><?= htmlspecialchars($recep['control']) ?></td>
              <td><?= date("d/m/Y", strtotime($recep['fecha_recepcion'])) ?></td>
              <td style="text-align: left;"><?= htmlspecialchars($recep['proveedor_nombre']) ?></td>
              <td>
                <?php if ($recep['status'] === 'APROBADO'): ?>
                    <span class="badge bg-success">APROBADO</span>
                <?php else: ?>
                    <span class="badge bg-warning text-dark"><?= htmlspecialchars($recep['status']) ?></span>
                <?php endif; ?>
              </td>
              <td style="font-weight: bold;"><?= number_format($recep['total_peso_neto'], 2, ',', '.') ?></td>
              <td><?= htmlspecialchars($recep['aprobado_por']) ?? 'N/A' ?></td>
              <td>
                <?= $recep['fecha_aprobacion'] ? date("d/m/Y H:i", strtotime($recep['fecha_aprobacion'])) : 'N/A' ?>
              </td>
              <td>
                <a href="ticket_pesaje.php?id=<?= $recep['id_recepcion'] ?>" target="_blank" class="boton" style="font-size: 0.9em; padding: 5px 10px; background-color:#1976d2;">
                   <i class="bi bi-printer"></i> Imprimir
                </a>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
        <tfoot>
            <tr class="total-row">
                <td colspan="4" style="text-align: right; font-weight: bold;">TOTAL GENERAL DEL REPORTE:</td>
                <td><?= number_format($total_reporte, 2, ',', '.') ?></td>
                <td colspan="3"></td>
            </tr>
        </tfoot>
      </table>
  <?php else: ?>
      <div class="alert alert-warning mt-4 text-center">
          <i class="bi bi-exclamation-triangle"></i> No se encontraron recepciones con los filtros seleccionados.
      </div>
  <?php endif; ?>

</div>

</body>
</html>