<?php
session_start();

// Validar sesión
if (!isset($_SESSION["id_usuario"])) {
    header("Location: login.php");
    exit();
}

require 'db.php';

$recepcion = null;
$detalle = [];
$mensaje = "";
$mensaje_exito = "";

/* ==========================================
   1. LÓGICA DE APROBACIÓN
   ========================================== */
if (isset($_POST['aprobar']) && isset($_POST['id_recepcion'])) {
    $id_recepcion = intval($_POST['id_recepcion']);
    $usuario_aprueba = $_SESSION['nombre']; 
    $fecha_actual = date("Y-m-d H:i:s");

    // 1. Obtener almacén y control de la cabecera (recepcion_general)
    $stmt_info = $conn->prepare("SELECT almacen, control FROM recepcion_general WHERE id_recepcion = ?");
    $stmt_info->bind_param("i", $id_recepcion);
    $stmt_info->execute();
    $res_info = $stmt_info->get_result()->fetch_assoc();
    $stmt_info->close();

    if ($res_info) {
        $almacen_destino = $res_info['almacen'];
        $control_referencia = $res_info['control'];

        // 2. Actualizar estatus a APROBADO
        $stmt = $conn->prepare("UPDATE recepcion_general 
            SET status = 'APROBADO', aprobado_por = ?, fecha_aprobacion = ?
            WHERE id_recepcion = ?");
        $stmt->bind_param("ssi", $usuario_aprueba, $fecha_actual, $id_recepcion);

        if ($stmt->execute()) {
            
            // 3. Obtener productos desde detalle_recepcion
            // SEGÚN TU IMAGEN: Los campos son codigo_producto, descripcion, presentacion, peso_neto
            $stmt_det = $conn->prepare("SELECT codigo_producto, descripcion, presentacion, peso_neto 
                                    FROM detalle_recepcion 
                                    WHERE id_recepcion = ?");
            $stmt_det->bind_param("i", $id_recepcion);
            $stmt_det->execute();
            $productos = $stmt_det->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt_det->close();

            // 4. Insertar en tabla stock
            // SEGÚN TU IMAGEN: Los campos destino son codigo, descripcion, presentacion, cantidad, almacen
            $stmt_stock = $conn->prepare("
                INSERT INTO stock (codigo, descripcion, presentacion, cantidad, almacen)
                VALUES (?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE cantidad = cantidad + VALUES(cantidad)
            ");

            foreach ($productos as $prod) {
                $stmt_stock->bind_param("sssds", 
                    $prod['codigo_producto'], // Va al campo 'codigo' del stock
                    $prod['descripcion'],     // Va al campo 'descripcion'
                    $prod['presentacion'],    // Va al campo 'presentacion'
                    $prod['peso_neto'],       // Va al campo 'cantidad' del stock
                    $almacen_destino          // Va al campo 'almacen'
                );
                
                if (!$stmt_stock->execute()) {
                    error_log("Error SQL Stock: " . $stmt_stock->error);
                }
            }
            $stmt_stock->close();

            $mensaje_exito = "✅ Recepción aprobada y stock actualizado correctamente.";
            
            // Recargar vista para ver el cambio de estatus
            $_POST['control'] = $control_referencia;

        } else {
            $mensaje = "❌ Error al actualizar el estatus: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $mensaje = "❌ Error crítico: No se encontró la recepción.";
    }
}

/* ==========================================
   2. LÓGICA DE BÚSQUEDA
   ========================================== */
if (isset($_POST['control']) || (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['control']))) {
    
    $control = trim($_POST['control']);

    if ($control !== "") {
        $stmt = $conn->prepare("
            SELECT rg.*, p.razon_social AS proveedor_nombre 
            FROM recepcion_general rg 
            LEFT JOIN proveedor p ON rg.rif_proveedor = p.cedula 
            WHERE rg.control = ?
        ");
        $stmt->bind_param("s", $control);
        $stmt->execute();
        $result = $stmt->get_result();
        $recepcion = $result->fetch_assoc();
        $stmt->close();

        if ($recepcion) {
            $stmt = $conn->prepare("
                SELECT descripcion, cantidad, peso_bruto, peso_tara, peso_neto 
                FROM detalle_recepcion 
                WHERE id_recepcion = ?
            ");
            $stmt->bind_param("i", $recepcion['id_recepcion']);
            $stmt->execute();
            $detalle = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $stmt->close();
        } else {
            if(empty($mensaje_exito)) {
                $mensaje = "No se encontró una recepción con ese número de control.";
            }
        }
    } else {
        $mensaje = "Debe ingresar un número de control.";
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Aprobar Recepción</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
  <style>
    body { font-family: 'Segoe UI', sans-serif; background-color: #f4fff6; padding: 30px; }
    .container { max-width: 1100px; margin: auto; background: #fff; padding: 25px; border-radius: 10px; box-shadow: 0 0 15px rgba(0,0,0,0.1); }
    h2 { text-align: center; color: #2e7d32; margin-bottom: 20px; }
    .boton { padding: 8px 15px; background-color: #2e7d32; color: white; border: none; border-radius: 5px; cursor: pointer; margin-left: 10px; text-decoration: none; display: inline-block;}
    .boton:hover { background-color: #1b5e20; color: white; }
    table { width: 100%; border-collapse: collapse; margin-top: 25px; }
    th, td { border: 1px solid #ccc; padding: 6px; text-align: center; }
    th { background-color: #e8f5e9; color: #2e7d32; }
    .acciones { margin-top: 20px; display: flex; gap: 15px; }
    .mensaje { margin-top: 15px; padding: 10px; background: #fff3cd; color: #856404; border: 1px solid #ffeeba; border-radius: 5px; }
  </style>
</head>
<body>
<div class="container">
  <h2>Buscar y Aprobar Recepción</h2>

 <form method="post" style="display: flex; align-items: center; gap: 10px;">
  <label for="control">Buscar por Control:</label>
  <input type="text" name="control" id="control" class="form-control" style="width: 300px;" placeholder="Ej: 0000007" value="<?= isset($_POST['control']) ? htmlspecialchars($_POST['control']) : '' ?>" required>
  <button type="submit" class="boton">Buscar</button>
  <a href="aprobar_recepciones.php" class="boton" style="background-color:#9e9e9e;">Limpiar</a>
  <a href="home.php" class="boton" style="background-color:#1976d2;">Volver</a>
</form>

  <?php if ($mensaje): ?>
    <div class="mensaje"><?= htmlspecialchars($mensaje) ?></div>
  <?php endif; ?>

  <?php if ($mensaje_exito): ?>
    <div class="alert alert-success mt-3"><?= htmlspecialchars($mensaje_exito) ?></div>
  <?php endif; ?>

  <?php if ($recepcion): ?>
    <hr>
    <h3>Recepción: <?= htmlspecialchars($recepcion['control']) ?></h3>
    <p><strong>Status:</strong>
      <?php if ($recepcion['status'] === 'APROBADO'): ?>
        <span class="badge bg-success">APROBADO</span>
      <?php else: ?>
        <span class="badge bg-warning text-dark"><?= htmlspecialchars($recepcion['status']) ?></span>
      <?php endif; ?>
    </p>

    <div class="row">
        <div class="col-md-6">
            <p><strong>Fecha:</strong> <?= date("d/m/Y", strtotime($recepcion['fecha_recepcion'])) ?></p>
            <p><strong>Proveedor:</strong> <?= htmlspecialchars($recepcion['proveedor_nombre']) ?></p>
            <p><strong>Guía SADA:</strong> <?= htmlspecialchars($recepcion['guia_sada']) ?></p>
            <p><strong>Almacén Destino:</strong> <?= htmlspecialchars($recepcion['almacen']) ?></p>
        </div>
        <div class="col-md-6">
            <p><strong>Vehículo:</strong> <?= htmlspecialchars($recepcion['vehiculo']) ?></p>
            <p><strong>Chofer:</strong> <?= htmlspecialchars($recepcion['chofer']) ?></p>
            <p><strong>Placa Chuto:</strong> <?= htmlspecialchars($recepcion['placa_chuto']) ?></p>
            <p><strong>Placa Batea:</strong> <?= htmlspecialchars($recepcion['placa_batea']) ?></p>
        </div>
    </div>

    <table>
      <thead>
        <tr>
          <th>Descripción</th>
          <th>Cantidad</th>
          <th>Peso Bruto</th>
          <th>Peso Tara</th>
          <th>Peso Neto</th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($detalle as $row): ?>
          <tr>
            <td><?= htmlspecialchars($row['descripcion']) ?></td>
            <td><?= number_format($row['cantidad'], 2, ',', '.') ?></td>
            <td><?= number_format($row['peso_bruto'], 2, ',', '.') ?></td>
            <td><?= number_format($row['peso_tara'], 2, ',', '.') ?></td>
            <td><?= number_format($row['peso_neto'], 2, ',', '.') ?></td>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>

    <div class="acciones">
      <?php if ($recepcion['status'] !== 'APROBADO'): ?>
        <button type="button" class="boton" data-bs-toggle="modal" data-bs-target="#modalConfirmar">Aprobar Recepción</button>
      <?php else: ?>
        <button type="button" class="boton" style="background-color:gray; cursor:not-allowed;" disabled>Recepción Aprobada</button>
      <?php endif; ?>

      <a href="ticket_pesaje.php?id=<?= $recepcion['id_recepcion'] ?>" target="_blank" class="boton" style="background-color:#1976d2;">Imprimir Ticket</a>
    </div>
  <?php endif; ?>
</div>

<div class="modal fade" id="modalConfirmar" tabindex="-1" aria-labelledby="modalConfirmarLabel" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <form method="post">
        <div class="modal-header">
          <h5 class="modal-title" id="modalConfirmarLabel">Confirmar Aprobación</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>
        <div class="modal-body">
          ¿Está seguro que desea aprobar esta recepción? <br>
          <small class="text-danger">Esta acción sumará el inventario al almacén <b><?= htmlspecialchars($recepcion['almacen']) ?></b>.</small>
        </div>
        <div class="modal-footer">
          <input type="hidden" name="id_recepcion" value="<?= isset($recepcion['id_recepcion']) ? $recepcion['id_recepcion'] : '' ?>">
          <input type="hidden" name="aprobar" value="1">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
          <button type="submit" class="btn btn-success">Sí, Aprobar</button>
        </div>
      </form>
    </div>
  </div>
</div>

</body>
</html>